/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * xitk X11 functions
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xitk.h"
#include "xitk_x11.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/types.h>
/*
#include <sys/wait.h>
#include <sys/utsname.h>
*/
#include <string.h>
#include <strings.h>
#include <unistd.h>

#include <stdint.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/keysym.h>
#include <X11/Xatom.h>
#ifdef HAVE_XF86VIDMODE
#  include <X11/extensions/xf86vmode.h>
#endif
#ifdef HAVE_X11_EXTENSIONS_XSHM_H
#  include <X11/extensions/XShm.h>
#endif
#ifdef HAVE_XV
#  include <X11/extensions/Xvlib.h>
#endif /* !HAVE_XV */

#include <xine/xineutils.h> /* xine_get_homedir() */
#include <xine/sorted_array.h>

#include "_xitk.h" /* xitk_get_bool_value */
#include "_backend.h"

#include "xitkintl.h"

/*
*
*/

int xitk_x11_parse_geometry (const char *geomstr, xitk_rect_t *r) {
  int xoff, yoff, ret;
  unsigned int width, height;

  if (!geomstr || !r)
    return 0;

  ret = XParseGeometry (geomstr, &xoff, &yoff, &width, &height);
  if (!ret)
    return 0;
  if (ret & XValue)
    r->x = xoff;
  if (ret & YValue)
    r->y = yoff;
  if (ret & WidthValue)
    r->width = width;
  if (ret & HeightValue)
    r->height = height;
  return 1;
}

void xitk_x11_xrm_parse(const char *xrm_class_name,
                        char **geometry, int *borderless,
                        char **prefered_visual, int *install_colormap) {
  char          buf[2048], *p, *e = buf + sizeof (buf) - 32;
  Display      *display;
  const char   *environment;
  char         *str_type;
  XrmDatabase   rmdb = NULL, temp_rmdb;
  XrmValue      value;

  XrmInitialize();

  if((display = XOpenDisplay((getenv("DISPLAY")))) == NULL)
    return;

  p = buf;
  memcpy (p, "/usr/lib/X11/app-defaults/", 26); p += 26;
  strlcpy (p, xrm_class_name, e - p);
  temp_rmdb = XrmGetFileDatabase (buf);
  if (temp_rmdb)
    XrmMergeDatabases (temp_rmdb, &rmdb);

  /* server rmdb */
  p = XResourceManagerString (display);
  if (p) {
    temp_rmdb = XrmGetStringDatabase (p);
  } else {
    p = buf;
    p += strlcpy (p, xine_get_homedir (), e - p);
    if (p > e)
      p = e;
    memcpy (p, "/.Xdefaults", 12); /* p += 12; */
    temp_rmdb = XrmGetFileDatabase (buf);
  }
  if (temp_rmdb)
    XrmMergeDatabases (temp_rmdb, &rmdb);

  /* home rmdb */
  environment = getenv ("XENVIRONMENT");
  if (!environment) {
    p = buf;
    p += strlcpy (p, xine_get_homedir (), e - p);
    if (p > e)
      p = e;
    memcpy (p, "/.Xdefaults-", 12); p += 12;
    if (gethostname (p, e - p) < 0)
      goto fail;
    *e = 0;
    environment = buf;
  }
  temp_rmdb = XrmGetFileDatabase (environment);
  if (temp_rmdb)
    XrmMergeDatabases (temp_rmdb, &rmdb);

  if (geometry) {
    if (XrmGetResource (rmdb, "xine.geometry", "xine.Geometry", &str_type, &value) == True)
      *geometry = strdup ((const char *)value.addr);
  }

  if (borderless) {
    if (XrmGetResource (rmdb, "xine.border", "xine.Border", &str_type, &value) == True)
      *borderless = !xitk_get_bool_value ((char *)value.addr);
  }

  if (prefered_visual) {
    if (XrmGetResource (rmdb, "xine.visual", "xine.Visual", &str_type, &value) == True)
      *prefered_visual = strdup ((const char *)value.addr);
  }

  if (install_colormap) {
    if(XrmGetResource (rmdb, "xine.colormap", "xine.Colormap", &str_type, &value) == True)
      *install_colormap = !xitk_get_bool_value ((char *)value.addr);
  }

  XrmDestroyDatabase(rmdb);

 fail:
  XCloseDisplay(display);
}

static int parse_visual(VisualID *vid, int *vclass, const char *visual_str) {
  int ret = 0;
  unsigned int visual = 0;

  if (sscanf(visual_str, "%x", &visual) == 1) {
    *vid = visual;
    ret = 1;
  }
  else if (strcasecmp(visual_str, "StaticGray") == 0) {
    *vclass = StaticGray;
    ret = 1;
  }
  else if (strcasecmp(visual_str, "GrayScale") == 0) {
    *vclass = GrayScale;
    ret = 1;
  }
  else if (strcasecmp(visual_str, "StaticColor") == 0) {
    *vclass = StaticColor;
    ret = 1;
  }
  else if (strcasecmp(visual_str, "PseudoColor") == 0) {
    *vclass = PseudoColor;
    ret = 1;
  }
  else if (strcasecmp(visual_str, "TrueColor") == 0) {
    *vclass = TrueColor;
    ret = 1;
  }
  else if (strcasecmp(visual_str, "DirectColor") == 0) {
    *vclass = DirectColor;
    ret = 1;
  }

  return ret;
}

void xitk_x11_find_visual (Display *display, int screen, const char *prefered_visual,
  Visual **visual_out, int *depth_out) {
  XWindowAttributes  attribs;
  XVisualInfo        vinfo_tmpl;
  XVisualInfo       *vinfo;
  int                num_visuals;
  int                depth = 0;
  Visual            *visual = NULL;
  VisualID           prefered_visual_id = None;
  int                prefered_visual_class = -1;

  if (screen == -1)
    screen = DefaultScreen (display);

  if (prefered_visual) {
    if (!parse_visual(&prefered_visual_id, &prefered_visual_class, prefered_visual)) {
      fprintf(stderr, "error parsing visual '%s'\n", prefered_visual);
    }
  }

  if (prefered_visual_id == None) {
    /*
     * List all available TrueColor visuals, pick the best one for xine.
     * We prefer visuals of depth 15/16 (fast).  Depth 24/32 may be OK,
     * but could be slow.
     */
    vinfo_tmpl.screen = screen;
    vinfo_tmpl.class  = (prefered_visual_class != -1 ? prefered_visual_class : TrueColor);
    vinfo = XGetVisualInfo(display, VisualScreenMask | VisualClassMask,
                           &vinfo_tmpl, &num_visuals);
    if (vinfo != NULL) {
      int i, pref;
      int best_visual_index = -1;
      int best_visual = -1;

      for (i = 0; i < num_visuals; i++) {
        if (vinfo[i].depth == 15 || vinfo[i].depth == 16)
          pref = 3;
        else if (vinfo[i].depth > 16)
          pref = 2;
        else
          pref = 1;

        if (pref > best_visual) {
          best_visual = pref;
          best_visual_index = i;
        }
      }

      if (best_visual_index != -1) {
        depth = vinfo[best_visual_index].depth;
        visual = vinfo[best_visual_index].visual;
      }

      XFree(vinfo);
    }
  } else {
    /*
     * Use the visual specified by the user.
     */
    vinfo_tmpl.visualid = prefered_visual_id;
    vinfo = XGetVisualInfo(display, VisualIDMask, &vinfo_tmpl, &num_visuals);
    if (vinfo == NULL) {
      fprintf(stderr, _("gui_main: selected visual %#lx does not exist, trying default visual\n"),
             (long) prefered_visual_id);
    } else {
      depth = vinfo[0].depth;
      visual = vinfo[0].visual;
      XFree(vinfo);
    }
  }

  if (depth == 0) {
    XVisualInfo vinfo;

    XGetWindowAttributes(display, (DefaultRootWindow(display)), &attribs);

    depth = attribs.depth;

    if (XMatchVisualInfo(display, screen, depth, TrueColor, &vinfo)) {
      visual = vinfo.visual;
    } else {
      fprintf (stderr, _("gui_main: couldn't find true color visual.\n"));

      depth = DefaultDepth (display, screen);
      visual = DefaultVisual (display, screen);
    }
  }

  if (depth_out)
    *depth_out = depth;
  if (visual_out)
    *visual_out = visual;
}

/*
 *
 */

void xitk_x11_select_visual(xitk_t *xitk, Visual *gui_visual) {
  xitk_be_display_t *d = xitk->d;
  if (d->set_visual)
    d->set_visual(d, gui_visual);
}


/*
 *
 */

xitk_window_t *xitk_x11_wrap_window(xitk_t *xitk, xitk_be_display_t *be_display, Window window) {
  if (xitk && (!xitk->be || xitk->be->type != XITK_BE_TYPE_X11)) {
    XITK_WARNING("Trying to wrap X11 window to non-X11 backend %d\n", xitk->be->type);
    return NULL;
  }
  if (be_display && be_display->type != XITK_BE_TYPE_X11) {
    XITK_WARNING("Tried to wrap X11 window to non-X11 backend %d\n", be_display->type);
    return NULL;
  }
  if (window == None)
    return NULL;
  return xitk_window_wrap_native_window (xitk, be_display, (uintptr_t)window);
}

/*
 *
 */

#ifdef HAVE_XF86VIDMODE
void xitk_x11_modelines_init(Display *display, xitk_x11_modelines_t *m) {

  int dummy_query_event, dummy_query_error;
  if (XF86VidModeQueryExtension (display, &dummy_query_event, &dummy_query_error)) {
    XF86VidModeModeInfo* XF86_modelines_swap;
    int                  mode, major, minor, sort_x, sort_y;
    int                  screen = XDefaultScreen(display);

    XF86VidModeQueryVersion (display, &major, &minor);
    printf (_("XF86VidMode Extension (%d.%d) detected, trying to use it.\n"), major, minor);

    if (XF86VidModeGetAllModeLines (display, screen, &m->count, &m->info)) {
      printf (_("XF86VidMode Extension: %d modelines found.\n"), m->count);

      /* first, kick off unsupported modes */
      for (mode = 1; mode < m->count; mode++) {

        if (!XF86VidModeValidateModeLine (display, screen, m->info[mode])) {
          int wrong_mode;

          printf(_("XF86VidModeModeLine %dx%d isn't valid: discarded.\n"),
                 m->info[mode]->hdisplay,
                 m->info[mode]->vdisplay);

          for (wrong_mode = mode; wrong_mode < m->count; wrong_mode++)
            m->info[wrong_mode] = m->info[wrong_mode + 1];

          m->info[wrong_mode] = NULL;
          m->count--;
          mode--;
        }
      }

      /*
       * sorting modelines, skipping first entry because it is the current
       * modeline in use - this is important so we know to which modeline
       * we have to switch to when toggling fullscreen mode.
       */
      for (sort_x = 1; sort_x < m->count; sort_x++) {

        for (sort_y = sort_x+1; sort_y < m->count; sort_y++) {

          if (m->info[sort_x]->hdisplay > m->info[sort_y]->hdisplay) {
            XF86_modelines_swap = m->info[sort_y];
            m->info[sort_y] = m->info[sort_x];
            m->info[sort_x] = XF86_modelines_swap;
          }
        }
      }
    } else {
      m->count = 0;
      printf(_("XF86VidMode Extension: could not get list of available modelines. Failed.\n"));
    }
  } else {
    printf(_("XF86VidMode Extension: initialization failed, not using it.\n"));
  }
}
#endif /* HAVE_XF86VIDMODE */

#ifdef HAVE_XF86VIDMODE
void xitk_x11_modelines_adjust(Display *display, Window window, xitk_x11_modelines_t *m, int width, int height) {
  int search = 0;

  if (m->count < 1)
    return;

  if (width > 0 && height > 0) {
    /* skipping first entry because it is the current modeline */
    for (search = 1; search < m->count; search++) {
      if (m->info[search]->hdisplay >= width)
        break;
    }
  }

  /*
   * in case we have a request for a resolution higher than any available
   * ones we take the highest currently available.
   */
  if (search >= m->count)
    search = 0;

  /* just switching to a different modeline if necessary */
  if (m->current != search) {
    if (XF86VidModeSwitchToMode (display, XDefaultScreen (display), m->info[search])) {
      m->current = search;

      // TODO
      /*
       * just in case the mouse pointer is off the visible area, move it
       * to the middle of the video window
       */
      XWarpPointer (display, None, window,
                    0, 0, 0, 0, m->info[search]->hdisplay / 2, m->info[search]->vdisplay / 2);

      XF86VidModeSetViewPort (display, XDefaultScreen (display), 0, 0);

    } else {
      //gui_msg (vwin->gui, XUI_MSG_ERROR, _("XF86VidMode Extension: modeline switching failed.\n"));
    }
  }
}
#endif /* HAVE_XF86VIDMODE */

#ifdef HAVE_XF86VIDMODE
void xitk_x11_modelines_reset(Display *display, xitk_x11_modelines_t *m) {
  /*
   * toggling from fullscreen to window mode - time to switch back to
   * the original modeline
   */

  if (m->count > 1 && m->current) {
    XF86VidModeSwitchToMode (display, XDefaultScreen (display), m->info[0]);
    XF86VidModeSetViewPort (display, XDefaultScreen (display), 0, 0);
    m->current = 0;
  }
}
#endif /* HAVE_XF86VIDMODE */

#ifdef HAVE_XF86VIDMODE
void xitk_x11_modelines_shutdown(Display *display, xitk_x11_modelines_t *m) {
  if (m->info) {
    xitk_x11_modelines_reset(display, m);
    XFree(m->info);
    m->info = NULL;
  }
}
#endif /* HAVE_XF86VIDMODE */

/* Code hardly based on xdyinfo.c, part of XFree86
 * Copyright 1988, 1998  The Open Group
 * Author:  Jim Fulton, MIT X Consortium
 * and xvinfo.c (no informations found) */

static int _sort_strcmp (const void *a, const void *b) {
  const char * const *d = (const char * const *)a;
  const char * const *e = (const char * const *)b;

  return strcmp (*d, *e);
}

void xitk_x11_dump_info (Display *display, int screen, int complete) {
  char                 dummybuf[40];
  const char          *cp, *svend;
  int                  i, n;
  long                 req_size;
  XPixmapFormatValues *pmf;
  Window               focuswin;
  int                  focusrevert, scr;
  char               **extlist;
  int                  nscreens, have_xv = 0;
  unsigned int         ver, rev, eventB, reqB, errorB;
  char                 line[320], *p;

  nscreens = XScreenCount (display);
  svend = XServerVendor (display);

  printf ("   Display Name:          %s,\n"
          "   XServer Vendor:        %s,\n"
          "   Protocol Version:      %d, Revision: %d,\n"
          "   Available Screen(s):   %d,\n"
          "   Default screen number: %d,\n"
          "   Using screen:          %d,\n"
          "   Depth:                 %d,\n",
          DisplayString (display),
          svend,
          XProtocolVersion (display), XProtocolRevision (display),
          nscreens,
          DefaultScreen (display),
          screen,
          XDisplayPlanes (display, screen));

#ifdef HAVE_SHM
  if(XShmQueryExtension(display)) {
    int major, minor, ignore;
    Bool       pixmaps;

    if(XQueryExtension(display, "MIT-SHM", &ignore, &ignore, &ignore)) {
      if(XShmQueryVersion(display, &major, &minor, &pixmaps ) == True) {
	printf("   XShmQueryVersion:      %d.%d,\n", major, minor);
      }
    }
  }
#endif

  if (!complete)
    return;

  if (strstr (svend, "XFree86")) {
    int vendrel = XVendorRelease (display);

    printf ("   XFree86 version: ");
    if (vendrel < 336) {
      /* vendrel was set incorrectly for 3.3.4 and 3.3.5, so handle those cases here. */
      printf ("      %d.%d.%d", vendrel / 100, (vendrel / 10) % 10, vendrel % 10);
    } else if (vendrel < 3900) {
      /* 3.3.x versions, other than the exceptions handled above */
      printf ("      %d.%d", vendrel / 1000, (vendrel /  100) % 10);
      if (((vendrel / 10) % 10) || (vendrel % 10)) {
        printf (".%d", (vendrel / 10) % 10);
        if (vendrel % 10) {
          printf (".%d", vendrel % 10);
        }
      }
    } else if (vendrel < 40000000) {
      /* 4.0.x versions */
      printf ("      %d.%d", vendrel / 1000, (vendrel /   10) % 10);
      if (vendrel % 10) {
        printf (".%d", vendrel % 10);
      }
    } else {
      /* post-4.0.x */
      printf ("      %d.%d.%d", vendrel / 10000000, (vendrel /   100000) % 100, (vendrel /     1000) % 100);
      if (vendrel % 1000) {
        printf(".%d", vendrel % 1000);
      }
    }
    printf ("\n");
  }

  req_size = XExtendedMaxRequestSize (display);
  if (!req_size)
    req_size = XMaxRequestSize (display);
  printf ("   Maximum request size:  %ld bytes,\n"
          "   Motion buffer size:    %ld,\n",
          req_size * 4,
          XDisplayMotionBufferSize (display));

  switch (BitmapBitOrder (display)) {
    case LSBFirst:
      cp = "LSBFirst";
      break;
    case MSBFirst:
      cp = "MSBFirst";
      break;
    default:
      snprintf (dummybuf, 40, "%s %d", "unknown order", BitmapBitOrder (display));
      cp = dummybuf;
      break;
  }
  printf ("   Bitmap unit:           %d,\n"
          "     Bit order:           %s,\n"
          "     Padding:             %d,\n",
          BitmapUnit (display),
          cp,
          BitmapPad (display));

  switch (ImageByteOrder (display)) {
    case LSBFirst:
      cp = "LSBFirst";
      break;
    case MSBFirst:
      cp = "MSBFirst";
      break;
    default:
      snprintf (dummybuf, 40, "%s %d", "unknown order", ImageByteOrder (display));
      cp = dummybuf;
      break;
  }
  printf ("   Image byte order:      %s,\n", cp);

  pmf = XListPixmapFormats (display, &n);
  printf ("   Number of supported pixmap formats: %d,\n", n);
  if (pmf) {
    printf ("   Supported pixmap formats:\n"
            "     Depth        Bits_per_pixel        Scanline_pad\n");
    for (i = 0; i < n; i++) {
      printf ("     %5d                 %5d               %5d\n",
        pmf[i].depth, pmf[i].bits_per_pixel, pmf[i].scanline_pad);
    }
    printf ("     -----------------------------------------------\n\n");
    XFree ((char *) pmf);
  }

    /*
     * when we get interfaces to the PixmapFormat stuff, insert code here
     */

  XGetInputFocus (display, &focuswin, &focusrevert);
  printf ("   Focus:                  ");
  switch (focuswin) {
    case PointerRoot:
      printf("PointerRoot\n");
      break;
    case None:
      printf("None\n");
      break;
    default:
      printf ("Window 0x%lx, revert to ", focuswin);
      switch (focusrevert) {
        case RevertToParent:
          printf ("Parent,\n");
          break;
        case RevertToNone:
          printf ("None,\n");
          break;
        case RevertToPointerRoot:
          printf ("PointerRoot,\n");
          break;
        default: /* should not happen */
          printf ("%d,\n", focusrevert);
      }
  }

  extlist = XListExtensions (display, &n);
  printf ("   Number of extensions:   %d\n", n);
  if (extlist) {
    size_t maxlen = 0;
    int i;

    for (i = 0; i < n; i++) {
      size_t li = strlen (extlist[i]);
      if (maxlen < li)
        maxlen = li;
    }
    maxlen += 2;

    qsort (extlist, n, sizeof (char *), _sort_strcmp);

    for (i = 0; i < n; i++) {
      size_t j = strlen (extlist[i]);
      int opcode, event, error;
      Bool retval = XQueryExtension (display, extlist[i], &opcode, &event, &error);

      p = line;
      memcpy (p, "     ", 5); p += 5;
      memcpy (p, extlist[i], j); p += j;
      j = maxlen - j;
      *p++ = ':';
      memset (p, ' ', j); p += j;
      if (retval == True) {
        memcpy (p, "[opcode: ", 9); p += 9;
        p += sprintf (p, "%d", opcode);
        if (event || error) {
          memcpy (p, ", base (", 8); p += 8;
          if (event) {
            memcpy (p, "event: ", 7); p += 7;
            p += sprintf (p, "%d", event);
            if (error)
              memcpy (p, ", ", 2), p += 2;
          }
          if (error) {
            memcpy (p, "error: ", 7); p += 7;
            p += sprintf (p, "%d", error);
          }
          *p++ = ')';
        }
        memcpy (p, "]\n", 2); p += 2;
      } else {
        memcpy (p, "not present.\n", 13); p += 13;
      }
      *p = 0;
      printf ("%s", line);
    }
    /* Don't free it */
    /* XFreeExtensionList (extlist); */
  }

#ifdef HAVE_XV
  if ((Success != XvQueryExtension (display, &ver, &rev, &reqB, &eventB, &errorB))) {
    printf ("   No X-Video Extension on %s\n", XDisplayName(NULL));
  } else {
    printf ("   X-Video Extension version: %u.%u\n", ver, rev);
    have_xv = 1;
  }
#endif /* !HAVE_XV */

  for (scr = 0; scr < nscreens; scr++) {
    Screen      *s = ScreenOfDisplay (display, scr);  /* opaque structure */
    double       xres, yres;
    int          ndepths = 0, *depths = NULL;
    unsigned int width, height;

    xres = ((((double)DisplayWidth (display, scr)) * 25.4) / (double)DisplayWidthMM(display, scr));
    yres = ((((double)DisplayHeight (display, scr)) * 25.4) / (double)DisplayHeightMM (display, scr));

    printf ("   Dimensions:             %dx%d pixels (%dx%d millimeters).\n"
            "   Resolution:             %dx%d dots per inch.\n",
      DisplayWidth (display, scr), DisplayHeight (display, scr),
      DisplayWidthMM (display, scr), DisplayHeightMM (display, scr),
      (int)(xres + 0.5), (int)(yres + 0.5));

    depths = XListDepths (display, scr, &ndepths);
    if (!depths)
      ndepths = 0;
    printf ("   Depths (%d):             ", ndepths);
    for (i = 0; i < ndepths - 1; i++)
      printf ("%d, ", depths[i]);
    for (; i < ndepths; i++)
      printf ("%d", depths[i]);
    printf ("\n");
    if (depths)
      XFree ((char *)depths);

    printf ("   Root window id:         0x%lx\n"
            "   Depth of root window:   %d plane%s\n"
            "   Number of colormaps:    min %d, max %d\n"
            "   Default colormap:       0x%lx\n"
            "   Default number of colormap cells:   %d\n"
            "   Preallocated pixels:    black %ld, white %ld\n"
            "   Options:                backing-store %s, save-unders %s\n",
            RootWindow (display, scr),
            DisplayPlanes (display, scr), DisplayPlanes (display, scr) == 1 ? "" : "s",
            MinCmapsOfScreen (s), MaxCmapsOfScreen (s),
            DefaultColormap (display, scr),
            DisplayCells (display, scr),
            BlackPixel (display, scr), WhitePixel (display, scr),
            (DoesBackingStore (s) == NotUseful) ? "no" : ((DoesBackingStore (s) == Always) ? "yes" : "when"),
            DoesSaveUnders (s) ? "yes" : "no");

    XQueryBestSize (display, CursorShape, RootWindow (display, scr), 65535, 65535, &width, &height);
    printf("   Largest cursor:         ");
    if ((width == 65535) && (height == 65535))
      printf ("unlimited\n");
    else
      printf ("%ux%u\n", width, height);

#ifdef HAVE_XV
    if (have_xv) {
      unsigned int          nencode, adaptor, nadaptors, k;
      int                   nattr, numImages;
      XvAdaptorInfo        *ainfo;
      XvAttribute          *attributes;
      XvEncodingInfo       *encodings;
      XvFormat             *format;
      XvImageFormatValues  *formats;

      printf ("   Xv infos:\n");

      XvQueryAdaptors (display, RootWindow(display, scr), &nadaptors, &ainfo);
      if (!nadaptors) {
        printf ("     No adaptors present.\n");
        continue;
      }

      for (adaptor = 0; adaptor < nadaptors; adaptor++) {
        char *t;
        p = line;
        p += sprintf (p,
          "    Adaptor #%u:         \"%s\"\n"
          "       Number of ports:   %li\n"
          "       Port base:         %li\n"
          "       Operations supported: ",
          adaptor, ainfo[adaptor].name,
          ainfo[adaptor].num_ports,
          ainfo[adaptor].base_id);
        t = p;
        if (ainfo[adaptor].type & XvInputMask) {
          if (ainfo[adaptor].type & XvVideoMask)
            memcpy (p, "PutVideo ", 9), p += 9;
          if (ainfo[adaptor].type & XvStillMask)
            memcpy (p, "PutStill ", 9), p += 9;
          if (ainfo[adaptor].type & XvImageMask)
            memcpy (p, "PutImage ", 9), p += 9;
        }
        if (ainfo[adaptor].type & XvOutputMask) {
          if (ainfo[adaptor].type & XvVideoMask)
            memcpy (p, "GetVideo ", 9), p += 9;
          if (ainfo[adaptor].type & XvStillMask)
            memcpy (p, "GetStill ", 9), p += 9;
          if (ainfo[adaptor].type & XvImageMask)
            memcpy (p, "GetImage ", 9), p += 9;
        }
        if (p == t)
          memcpy (p, "none ", 5), p += 5;
        *p++ = '\n';
        *p = 0;
        printf ("%s", line);

        format = ainfo[adaptor].formats;
        printf ("     Supported visuals:\n");
        {
          int last_depth = 0;
          unsigned int first_id = 0, last_id = 0;
          for (k = 0; k < ainfo[adaptor].num_formats; k++, format++) {
            if (((int)format->depth == last_depth) && (format->visual_id == last_id + 1)) {
              last_id++;
              if (k + 1 < ainfo[adaptor].num_formats)
                continue;
            }
            if (first_id) {
              if (last_id > first_id)
                printf ("       - Depth %d, visualIDs 0x%2x-0x%2x\n", last_depth, first_id, last_id);
              else
                printf ("       - Depth %d, visualID 0x%2x\n", last_depth, first_id);
            }
            last_depth = format->depth;
            first_id = last_id = format->visual_id;
          }
        }

        attributes = XvQueryPortAttributes (display, ainfo[adaptor].base_id, &nattr);
        if (attributes && nattr) {
          int j;
          printf ("     Number of attributes: %d\n", nattr);
          for (j = 0; j < nattr; j++)
            printf ("       - %s\n", attributes[j].name);
          XFree (attributes);
        } else {
          printf ("     No port attributes defined.\n");
        }

        XvQueryEncodings (display, ainfo[adaptor].base_id, &nencode, &encodings);
        if (encodings && nencode) {
          unsigned int n, ImageEncodings = 0;
          for (n = 0; n < nencode; n++) {
            if (!strcmp (encodings[n].name, "XV_IMAGE"))
              ImageEncodings++;
          }
          if (nencode - ImageEncodings) {
            printf ("     Number of encodings: %u\n", nencode - ImageEncodings);
            for (n = 0; n < nencode; n++) {
              if (strcmp (encodings[n].name, "XV_IMAGE")) {
                printf ("       - encoding ID #%li: \"%s\"\n"
                        "         size:             %li x %li\n"
                        "         rate:             %f\n",
                  encodings[n].encoding_id, encodings[n].name,
                  encodings[n].width, encodings[n].height,
                  (float)encodings[n].rate.numerator / (float)encodings[n].rate.denominator);
              }
            }
          }

          if (ImageEncodings && (ainfo[adaptor].type & XvImageMask)) {
            char imageName[5] = {0, 0, 0, 0, 0};
            unsigned int j;
            int n;

            for (j = 0; j < nencode; j++) {
              if (!strcmp (encodings[j].name, "XV_IMAGE")) {
                printf ("     Maximum XvImage size: %li x %li\n", encodings[j].width, encodings[j].height);
                break;
              }
            }

            formats = XvListImageFormats (display, ainfo[adaptor].base_id, &numImages);
            printf ("     Number of image formats: %d\n", numImages);
            for (n = 0; n < numImages; n++) {
              memcpy (imageName, &(formats[n].id), 4);
              printf ("       - Id: 0x%x", formats[n].id);
              if (isprint (imageName[0]) && isprint (imageName[1]) &&
                isprint (imageName[2]) && isprint (imageName[3])) {
                printf (" (%s):\n", imageName);
              } else {
                printf (":\n");
              }
              printf ("          Bits per pixel: %d\n"
                      "          Number of planes: %d\n"
                      "          Type: %s (%s)\n",
                formats[n].bits_per_pixel,
                formats[n].num_planes,
                (formats[n].type == XvRGB) ? "RGB" : "YUV",
                (formats[n].format == XvPacked) ? "packed" : "planar");
              if (formats[n].type == XvRGB) {
                printf ("          Depth: %d\n"
                        "          Red, green, blue masks: 0x%x, 0x%x, 0x%x\n",
                  formats[n].depth,
                  formats[n].red_mask, formats[n].green_mask, formats[n].blue_mask);
              }
            }
            if (formats)
              XFree (formats);
          }
          XvFreeEncodingInfo (encodings);
        }
        printf ("    End #%u.\n", adaptor);
      }
      XvFreeAdaptorInfo (ainfo);
    }
#endif /* !HAVE_XV */
  }
}
